#include <ctime>
#include <assert.h>
#include "ezwin.h"
#include "bitmap.h"
#include <cstring>
#include <iostream>

SimpleWindow w("Snake", 20.0, 14.0, Position(2.0, 2.0));

/* Variable Description:

snake      : array to represent snake
map        : array to represent the map on which the snake moves
level      : represents the level on which user is playing
subLevel   : represents the number of fruits eaten (length of the snake)
delay      : represents the time delay between calling the functions
xSnake, ySnake : represent the co-ordinates of the snake's head
dir        : represents the direction of the snake(initially right)
fruit cord : represents the array of the fruit
fruit_type : represents the type of the fruit
center     : center of the window
head       : head of the snake
body       : body of the snake
tail       : tail of the snake

*/ 

int snake[15][2];
int map[40][28];
int level=1, subLevel=0;
int delay=200;
int xSnake=0, ySnake=0;				
char dir = 'R';					
int xFIni=0, yFIni=0;
int fruitType[12];
Position center=w.GetCenter();
BitMap head(w);
BitMap body(w);
BitMap tail(w);
BitMap fruit(w);

/* Function Description:

prepLevel   : function to design the level
playgame    : function for the basic gameplay
move        : function to move the snake 
randomCood  : function to generate the random co-ordinates of the fruit
loadMap     : function to load the bitmap images of the level
plotFruit   : function to plot the fruit randomly
mouseClick  : function to be invoked when a mouse click event occurs
timerClick  : function to be invoked repeatedly after certain time
check       : function to check for head crash
check_fruit : function to check whether fruit lies on the wall or snake

*/

int prepLevel();
int playGame();
int move();
int turn();
int randomCood(int l);
int loadMap();
int plotFruit();
int mouseClick(const Position &p);
int timerClick();
int check();
int check_fruit();

// main program
int ApiMain()
{
	//initialization of the co-ordinates of the head, body and tail of the snake
	snake[1][0]=20;snake[1][1]=14;snake[0][0]=21;snake[0][1]=14;snake[2][0]=19;snake[2][1]=14;
	//opening the game window
	w.Open();
	assert(w.GetStatus()==WindowOpen);
	//locating the center
	center=w.GetCenter();

	//initialising map array and fruitType array
	for(int i=0; i<40; i++)
	{
		for(int j=0; j<28; j++)
			map[i][j]=0;
	}
	for(int i=0; i<12; i++)
		fruitType[i]=0;
	
	//loading the images of the head, body and tail
	head.Load("head.xpm");
	assert(head.GetStatus()==BitMapOkay);
	body.Load("body.xpm");
	assert(body.GetStatus()==BitMapOkay);
	tail.Load("tail.xpm");
	assert(tail.GetStatus()==BitMapOkay);
	
	//set the initial position of the images of the head, body and tail
	body.SetPosition(center);
	head.SetPosition(center + Position(.5,0));
	tail.SetPosition(center + Position(-.5,0));

	//drawing the images of the head, body and tail
	head.Draw();
	body.Draw();
	tail.Draw();

	//starting the game
	playGame();
	return 0;
}

int playGame()
{
	//preparing the level
	prepLevel();
	//taking the input from mouse
	w.SetMouseClickCallback(mouseClick);
	//starting the timer and calling "timerClick" regularly
	w.SetTimerCallback(timerClick);
	w.StartTimer(delay);
	return 0;
}

int mouseClick(const Position &p)
{
	//if the direction of the snake is up or down then to change it, following code works:
	if(dir=='U'|| dir=='D')
	{
		//if the mouse click is to the left of the snake, then turn left
		if(p.GetXDistance()*2<snake[0][0])
		{
			dir='L';
		}
		//if the mouse click is to the right of the snake, then turn right
		else if(p.GetXDistance()*2>snake[0][0])
		{
			dir='R';
		}
	}
	//if the direction of the snake is left or right then to change it, following code works:
	else
	{
		//if the mouse click is above the snake, then turn upwards
		if(p.GetYDistance()*2<snake[0][1])
		{
			dir='U';
		}
		//if the mouse click is below the snake, then turn downwards 
		else if(p.GetYDistance()*2>snake[0][1])
		{
			dir='D';
		}
	}
	//turn the snake in the new direction
	turn();
	//move the snake
	move();
	check();
	return 0;
}

int timerClick()
{
	//call the following functions repeatedly after a certain time interval
	move();
	check();
	return 0;
}

int check()
{
/*map[][]=1 means there is a wall on that position
  map[][]=-1 means there is a fruit on that position
*/
	xSnake=snake[0][0];
	ySnake=snake[0][1];
	
	//The snake collides with wall
	if(map[xSnake][ySnake]==1)
	{
		//level=11 means game over
		level=11;
		//restart the game
		playGame();
		return 0;
	}

	//The snake eats the fruit	
	else if(map[xSnake][ySnake]==-1)
	{
		//if the number of fruits eaten is less than 12
		if(subLevel<12)
		{
			//change the sublevel(that is the no. of fruits eaten)
			subLevel++;
			//plot the next fruit
			plotFruit();
			return 0;
		}

		//if number of the fruits eaten is 12 
		else
		{
			//go to the next level
			level++;
			//snake comes back to initial length
			subLevel=0;
			//start the next level
			playGame();
			return 0;
		} 
	}

	//The snake collides with the body
	else
	{
		for(int i=1; i<subLevel+3; i++)
		{
			//Co-ordinates of head and a body part become same
			if(snake[0][0]==snake[i][0] && snake[0][1]==snake[i][1])
			{
				//game over
				level=11;
				//game Over Menu
				playGame();
				return 0;
			}
		}
	}
					
}

int prepLevel()
{
	plotFruit();
	return 0;
}

int move()
{	
	//Store the initial co-ordinates of head
	int tempX=snake[0][0];
	int tempY=snake[0][1];
	
	switch(dir)
	{
		//if snake goes upwards
		case 'U' : snake[0][1]-=1;
		break;
		//if snake goes downwards
		case 'D' : snake[0][1]+=1;
		break;
		//if snake goes left
		case 'L' : snake[0][0]-=1;
		break;
		//if snake goes right
		case 'R' : snake[0][0]+=1;
		break;
	}

	//"for" loop to transfer the co-ordinates of a body part to the next body part
	for(int i=2;i<subLevel+3;i++)
	{
		snake[i][0]=snake[i-1][0];
		snake[i][1]=snake[i-1][1];
	}

	//transfer the initial co-ordinates of head to the next body part
	snake[1][0]=tempX;
	snake[1][1]=tempY;
	//cout<<snake[0][0]<<" "<<snake[1][0]<<" "<<snake[2][0]; (used for debugging while checking the  program)

	
	//Setting new position for image of head
	head.Erase();
	head.SetPosition(Position((float)(snake[0][0])/2,((float)snake[0][1])/2));
	head.Draw();
	
	//Setting new position for image of body
	body.Erase();
	for(int i=1; i<subLevel+2; i++)
	{
		body.SetPosition(Position((float)(snake[i][0])/2,(float)(snake[i][1])/2));
		body.Draw();
	}
	
	//Setting new position for image of tail
	tail.Erase();
	tail.SetPosition(Position((float)(snake[subLevel+2][0])/2,(float)(snake[subLevel+2][1])/2));
	tail.Draw();
	return 0;
}

int turn()
{
	//switches according to the newly assigned direction
	switch(dir)
	{
		//changes the image for head and tail as per the new direction
		case 'U' : head.Load("headU.xpm");
		tail.Load("tailU.xpm");
		break;
		case 'D' : head.Load("headD.xpm");
		tail.Load("tailD.xpm");
		break;
		case 'R' : head.Load("headR.xpm");
		tail.Load("tailR.xpm");
		break;
		case 'L' : head.Load("headL.xpm");
		tail.Load("tailL.xpm");
		break;
	}
	return 0;
}

int plotFruit()
{
	int xFruit,yFruit,flag, fType;
	do{	
		flag=0;

		//generating random co-ordinates for the fruit
    		xFruit=randomCood(40);
    		yFruit=randomCood(28);

    		//if the co-ordinates generated lie on wall or snake
    		//checking for wall
		if(map[xFruit][yFruit]==1)
		{
			flag=1;
		}
		//checking for snake
		else
		{
			for(int i=0; i<subLevel+3; i++)
			{
				if(snake[i][0]==xFruit && snake[i][1]==yFruit)
					flag=1;
			}
		}
	}while(flag==1);	//if flag=1, generate another pair of co-ordinates
	//setting initial map coods to 0
	map[xFIni][yFIni]=0;
	//setting new fruit co-ordinate to -1
	map[xFruit][yFruit]=-1;
	//setting xFIni and yFIni
	xFIni=xFruit;
	yFIni=yFruit;
	//random fruit type
	int fruitRand;
	//checking whether this type has already has been used
	do{
		fruitRand=randomCood(12);
		if(fruitType[fruitRand]==1)
		{
			flag=1;
		}
		else flag=0;
	}while(flag==1);
	fruitType[fruitRand]=1;
	//creating file name
	string imageF;
	if(fruitRand>9){fruitRand=randomCood(10);}
	string a="fruit";
	imageF=a+(char)(49+fruitRand)+".xpm";
	//loading new fruit
	fruit.Load(imageF);
	fruit.SetPosition(Position(xFruit,yFruit));
	fruit.Draw();
	return 0;
}  	


int randomCood(int l)
{
	//seed value dependent on time
	srand(time(NULL));
			
	//generating random co-ordinates			
  	return rand()%l;
}

int levelChange()
{
	//code7
}

int loadMap()
{
	//code8
}

int mapPlot()
{
	//code9
}

